﻿//Based on the Uber Advanced shader this variation adds the option to
//have textured colored speculars.

Shader "!FullControl/Uber/Uber Advanced + Textured Specular" 
{
	Properties 
	{
		//Diffuse Properties
		_MainColor ("Diffuse Tint Color", Color) = (1, 1, 1, 1)
		_MainTex ("Albedo (RGB) | SpecMask (A)", 2D) = "grey" {}
		
		//NormalMap Properties
		_BumpMap ("Normal Map", 2D) = "bump" {}
		
		//Specular Properties
		_SpecularColorMap ("Specular Color (RGB)", 2D) = "white" {}
		_SpecularTint ("Specular Tint", Color) = (1, 1, 1, 1)
		_SpecularPower ("Specular Power", Range (0.01, 5)) = 0.078125
		_SpecularIntensity ("Specular Intensity", float) = 0
		
		//RimLight Properties
		_RimLightColor ("RimLight Color", Color) = (1,1,1,1)
		_RimLightFalloff ("RimLight Fresnell", float) = 1 
		_RimLightIntensity ("RimLight Intensity", float) = 0
		
		//Reflection Properties
		_Cubemap ("Reflection Cubemap", CUBE) = "black" {}
		_ReflectionColor ("Reflection Color", Color) = (1,1,1,1)
		_ReflectionFalloff ("Reflection Fresnell", float) = 1 
		_ReflIntensity ("Reflection Intensity", float) = 0
		
		//Emission Properties
		_EmissionColor ("Emission Color", Color) = (0,0,0,0)
		_EmissionTex ("Emission Map", 2D) = "black" {}
		_EmissionInt ("Emission Intensity", float) = 0.0
	}
	
	SubShader 
	{
		Tags {"Queue"="Geometry"}
		
		LOD 200
		
		CGPROGRAM
		#pragma target 3.0
		#pragma surface surf BlinnPhongColorSpec
		
		struct MySurfaceOutput 
		{
	    half3 Albedo;
	    half3 Normal;
	    half3 Emission;
	    half Specular;
	    half3 GlossColor;
	    half Alpha;
		};	
		
		inline half4 LightingBlinnPhongColorSpec (MySurfaceOutput s, half3 lightDir, half3 viewDir, half atten)
		{
		  	half3 h = normalize (lightDir + viewDir);
		  	half diff = max (0, dot (s.Normal, lightDir));
		  	float nh = max (0, dot (s.Normal, h));
		  	float spec = pow (nh, s.Specular * 32.0);
			//float spec = pow (nh, s.Specular*128.0) * s.Gloss;
		  	
		  	half3 specCol = spec * s.GlossColor;
		  	half4 c;
		  	
		 	c.rgb = (s.Albedo * _LightColor0.rgb * diff + _LightColor0.rgb * specCol) * (atten * 2);
		  	c.a = s.Alpha;
		  	return c;
	  	}

		inline half4 LightingBlinnPhongColorSpec_PrePass (MySurfaceOutput s, half4 light)
		{
		    half3 spec = light.a * s.GlossColor;
		    half4 c;
		    c.rgb = (s.Albedo * light.rgb + light.rgb * spec);
		    c.a = s.Alpha + spec * _SpecColor.a;
		    return c;
	    }

		
		//Diffuse
		float4 _MainColor;
		sampler2D _MainTex;
		sampler2D _BumpMap;
		
		//Specular
		sampler2D _SpecularColorMap;			
		float4 _SpecularTint;
		float _SpecularPower;
		float _SpecularIntensity;
		
		//RimLight
		float4 _RimLightColor;
		float _RimLightFalloff;
		float _RimLightIntensity;
		
		//Reflection
		samplerCUBE _Cubemap;
		float4 _ReflectionColor;
		float _ReflIntensity;
		float _ReflectionFalloff;
		
		//Emission
		sampler2D _EmissionTex;
		float4 _EmissionColor;
		float _EmissionInt;

		struct Input 
		{
			float2 uv_MainTex;
			float3 worldRefl;
			float3 viewDir;
			INTERNAL_DATA
		};
		

		void surf (Input IN, inout MySurfaceOutput o) 
		{
			//Diffuse
			half4 difTxt = tex2D (_MainTex, IN.uv_MainTex);
			half3 normalMap = UnpackNormal(tex2D(_BumpMap, IN.uv_MainTex));
			o.Normal = normalMap;
			
			//RimLight
			half rim = 1.0 - saturate(dot(o.Normal, normalize(IN.viewDir)));
			half3 finalRim = _RimLightColor.rgb * pow (rim, _RimLightFalloff) * _RimLightIntensity;
			
			//Reflection
			rim = pow(rim, _ReflectionFalloff);
			half4 specTex = tex2D (_SpecularColorMap, IN.uv_MainTex);
			half3 finalRefl = texCUBE (_Cubemap, WorldReflectionVector (IN, o.Normal)).rgb * _ReflIntensity * rim * _ReflectionColor * specTex.rgb;
			
			//Emission
			half3 emissionTex = tex2D (_EmissionTex, IN.uv_MainTex);
			o.Emission = finalRefl * difTxt.a + finalRim +((emissionTex + _EmissionColor) * _EmissionInt);
			
			//Specular
			o.GlossColor = difTxt.a * specTex.rgb * _SpecularIntensity * _SpecularTint;
			o.Specular = _SpecularPower * (difTxt.a * 2);		
			
			//Albedo	
			o.Albedo = difTxt.rgb * _MainColor;
		}
		ENDCG
	}
	//Disabled during development
	FallBack "Diffuse"
}
